<?php
/**
 * class_stc_cache, provides cache classes for steam connect
 *
 * @author Disasterpiece
 * @author Andreas "Radon" Rudolph <radon@purgatory-labs.de>
 * @version 1.4.5rc3
 */

/**
 * Abstract class to offer easy integration of cache methods
 */
class stc_cache {

	// Singleton
	private static $instance = null;

	private static $classname = null;

	public static function getInstance() {
		if (self::$classname == null) self::$classname = stc_get_cache_classname();
		if (self::$instance == null && isset(self::$classname)) {
			self::$instance = new self::$classname();
			DEVDEBUG("Using stc_cache_class: ".self::$classname);
		}
		return self::$instance;
	}

	public static function resetInstance() {
		// Primarily for test purposes
		self::$instance = null;
		self::$classname = null;
	}
	// End singleton

	/**
	 * Tests if a specific cache entry is valid
	 * @param string identifier the unique identifier for the specific cache entry
	 * @return bool if the cache entry is valid
	 */
	public function isValid($identifier) {}

	/**
	 * Stores an entry in the cache with the provided unique identifier
	 * @param string identifier the unique identifier for the specific cache entry
	 * @param mixed payload the data to store
	 * @param bool is_serialized serialized state of the payload
	 * @return bool if the store process was successful
	 */
	public function store($identifier, $payload, $is_serialized) {}

	/**
	 * Retrieves an entry from the cache, or false if it wasnt found
	 * @param string identifier the unique identifier for the specific cache entry
	 * @return mixed the entry or false if it wasnt found
	 */
    public function fetch($identifier) {}

	/**
	 * Removes an entry from the cache
	 * @param string identifier the unique identifier for the specific cache entry
	 * @return bool if the remove process was successful
	 */
    public function remove($identifier) {}

	/**
	 * returns the timestamp when the cache object has been created
	 * @param string identifier the unique identifier for the specific cache entry
	 * @return double the unix timestamp or 0 if the cache entry hasnt been found
	 */
    public function getCacheTime($identifier) {}

	/**
	 * Removes all outdated cache entries from the cache
	 * @return void
	 */
    public function purge() {}

}

class stc_file_cache extends stc_cache {

	private $cachedir;

	private $cheapcache;

	protected function __construct() {
		global $vbulletin;

		$this->cachedir = $vbulletin->options['stc_cachedir'];
		if ($this->cachedir{strlen($this->cachedir)-1} != '/') $this->cachedir .= '/';
		$this->cheapcache = array();
	}

	public function isValid($identifier) {
		global $vbulletin;

		if (isset($this->cheapcache[$identifier]))
			return true;

		$cachefile = $this->getCachePath($identifier);
		return (boolean)(file_exists($cachefile) && is_readable($cachefile) && (time() - $vbulletin->options['stc_cachetime'] <= $this->getCacheTime($identifier)));
	}

	public function store($identifier, $payload, $is_serialized = false) {
		$cachefile = $this->getCachePath($identifier);
		$res = file_put_contents($cachefile, $is_serialized ? $payload : serialize($payload));
		$this->cheapcache[$identifier] = $is_serialized ? unserialize($payload) : $payload;
		return $res;
	}

	public function fetch($identifier) {
		if (isset($this->cheapcache[$identifier]))
			return $this->cheapcache[$identifier];

		if (!$this->isValid($identifier)) return false;

		$cachefile = $this->getCachePath($identifier);

		$ret = unserialize(file_get_contents($cachefile));
		if (is_array($ret) && !empty($ret)) {
			$this->cheapcache[$identifier] = $ret;
			return $ret;
		} else {
			stc_log_action("malformed cachefile: $cachefile - deleted");
			$this->remove($identifier);
		}
		return false;
	}

	public function remove($identifier) {
		return file_exists($cachefile) && is_readable($cachefile) ? unlink($this->getCachePath($identifier)) : false;
	}

	public function getCacheTime($identifier) {
		$cachefile = $this->getCachePath($identifier);
		return file_exists($cachefile) ? filemtime($cachefile) : 0;
	}

	public function purge() {
		global $vbulletin;

       	// Clear outdated cache files
		if ($dirh = opendir($vbulletin->options['stc_cachedir'])) {
		    while (false !== ($entry = readdir($dirh))) {
				$path = $vbulletin->options['stc_cachedir'] . '/' . $entry;
				if (substr($entry, 0, 4) == 'stc_' && file_exists($path) && is_readable($path)) {
					if (time() - $vbulletin->options['stc_cachetime'] > filemtime($path)) {
						echo "deleting $entry<br>\n";
						unlink($path);
					} else {
						echo "file $entry still valid.<br>\n";
					}
		        } else {
					echo "omitting: $entry<br>\n";
				}
		    }
		    closedir($dirh);
		} else {
			echo "Could not create a dir handler for the cache dir.<br>\n";
		}
	}

	protected function getCachePath($name) {
		return $this->cachedir . "stc_$name.cached";
	}

}

class stc_memcached_cache extends stc_cache {
	
	private $memcache;

	private $cheapcache;

	protected function __construct() {
		global $vbulletin;

		$this->cheapcache = array();

		$servers = explode("\n", $vbulletin->options['stc_memcached_connection']);
        $url_schema = parse_url(trim($servers[0]));
		$this->memcache = new Memcache;
		$this->memcache->connect($url_schema['host'], $url_schema['port']);

		if (!$this->memcache) {
			trigger_error("Could not connect to memcache server @ $url_schema[host]:$url_schema[host]");
			return;
		}

		// Add additional servers if provided
		for ($i = 1; $i < count($servers); $i++) {
			if (!empty($servers[$i])) {
		        $url_schema = parse_url(trim($servers[$i]));
		        $this->memcache->addServer($url_schema['host'], $url_schema['port']);
			}
		}

	}

	public function isValid($identifier) {
		global $vbulletin;

		if (isset($this->cheapcache[$identifier]))
			return true;

		$obj = $this->memcache->get($identifier);
		return (boolean)(isset($obj) && (time() - $vbulletin->options['stc_cachetime'] <= $this->getCacheTime($identifier)));
	}

	public function store($identifier, $payload, $is_serialized = false) {
		global $vbulletin;

		$this->cheapcache[$identifier] = $is_serialized ? unserialize($payload) : $payload;
		$new_obj = new store_obj();
		$new_obj->time = time();
		$new_obj->data = $is_serialized ? unserialize($payload) : $payload;
		return $this->memcache->set($identifier, $new_obj, false, $vbulletin->options['stc_cachetime']);
	}

	public function fetch($identifier) {
		if (isset($this->cheapcache[$identifier]))
			return $this->cheapcache[$identifier];

		$obj = $this->memcache->get($identifier);
		return ($obj && get_class($obj) == 'store_obj' ? $obj->data : false);
	}

	public function remove($identifier) {
		return $this->memcache->delete($identifier);
	}

	public function getCacheTime($identifier) {
		$obj = $this->memcache->get($identifier);
		return ($obj ? $obj->time : 0);
	}

	public function purge() {
		return $this->memcache->flush();
	}

}

// Small struct so we can store creation time
class store_obj {
	public $time;
	public $data;
}

/**
 * Returns the class name of the cache to use, depending on the vbulletin setting
 */
function stc_get_cache_classname() {
	global $vbulletin;
	switch ($vbulletin->options['stc_cache_classname']) {

	case 'stc_file_cache':
		return 'stc_file_cache';

	case 'stc_memcached_cache':
		return 'stc_memcached_cache';

	case 'stc_xcache_cache':
		return 'stc_xcache_cache';

	case 'stc_datastore_cache':
		return 'stc_datastore_cache';
		
	default:
		trigger_error('functions_steamconnect.stc_get_cache_classname: Invalid classname: '.$vbulletin->options['stc_cache_classname']);
		return 'stc_cache';

	}
}

?>