<?php
/**
 * functions_steamconnect, provides handy functions in vbulletin context for usage with the steam connector plugin and other deriving mods
 *
 * @author Disasterpiece
 * @author Andreas "Radon" Rudolph <radon@purgatory-labs.de>
 * @version 1.4.5rc3
 */
 
require_once DIR . '/includes/class_stc_cache.php';

$stc_loghandle = NULL;

/**
 * returns the steam id from the given userinfo or the currently logged in user if the parameter is null
 * @param array $user_info the user info from where the steamid should be retrieved. Function defaults to logged in user if this param is null
 * @return string|boolean false if no steam id was found, otherwise the steam id
 */
function get_user_steamid($user_info=null) {
	global $vbulletin;

	if (!isset($user_info)) $user_info = &$vbulletin->userinfo;

	if (!isset($user_info['steam_link']) || empty($user_info['steam_link']))
	{
		return false;
	}
	else
	{
		list($steam_id,$x,$y) = explode('|',$user_info['steam_link']);
		return $steam_id;
	}
}

/**
 * returns weather the user from the userinfo uses a valid forum account or was created by the quick registration, therefore not using an email or password.
 * @param array $user_info the user info from where the info should be retrieved. Function defaults to logged in user if this param is null
 * @return boolean true if the userinfo shows, that the user doesn't have an email or a password specified, false otherwise.
 */
function is_steamonly_user($user_info=null) {
	global $vbulletin;

	if (!isset($user_info)) $user_info = &$vbulletin->userinfo;

	if (!isset($user_info['steam_link']) || empty($user_info['steam_link']))
	{
		return false;
	}
	else
	{
		list($x,$no_email,$no_passwd) = explode('|',$user_info['steam_link']);
		return ($no_email==1||$no_passwd==1);
	}
}

/**
 * Queries the Steam API for a profile summary and returns the result. returns false if the query was unsuccessfull.
 * @param string $steam_id the steamid used to specify which steam profile to query
 * @param string $api_key the Steam Web API key which provides access to use the actual Steam API
 * @param array $additional_fields can be used to add additional fields to the steam query other than defined in the whitelist
 * @return array|boolean array with contents of the specified fields from the query or false if the query was unsuccessfull because the steam id wasn't found or the server didn't respond.
 */
function fetch_steam_info($steam_id, $api_key, $additional_fields=array()) {
	global $vbulletin;
	$stc_cache = stc_cache::getInstance();

	if ($steam_id === false)
		return false;
	elseif (empty($steam_id)) {
		stc_log_action("fetch_steam_info called with empty steam_id!");
		return false;
	}

	$time = microtime(1);

	if (strpos($steam_id, '|')>0) {
		list($steam_id,$x,$y) = explode('|',$steam_id);
	}

	$cached_data = $stc_cache->fetch($steam_id);
	if ($cached_data) {
		DEVDEBUG("stc_profiling[] = Time: ".(microtime(1)-$time).", ret: cache");
		return $cached_data;
	}

	$url = sprintf("http://api.steampowered.com/ISteamUser/GetPlayerSummaries/v0002/?key=%s&steamids=%s",
		$api_key,
		$steam_id);

	$curl_time = microtime(1);
	// Needs curl to actually be able to pull data from steam api because of the special response-behaviour.
    $ch = curl_init();
    curl_setopt( $ch, CURLOPT_URL, $url );
    @curl_setopt( $ch, CURLOPT_FOLLOWLOCATION, true );
    curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
    curl_setopt( $ch, CURLOPT_AUTOREFERER, true );
    curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, false );
    curl_setopt( $ch, CURLOPT_MAXREDIRS, 5 );
    curl_setopt( $ch, CURLOPT_CONNECTTIMEOUT, $vbulletin->options['stc_curl_timeout'] ); #rc1: helps with page delay if steam is down
    $raw_json = curl_exec( $ch );
    $http_response_header = curl_getinfo( $ch );
	$curl_errno = curl_errno($ch);
	$curl_error = curl_error($ch);
	curl_close($ch);

	$curl_time = microtime(1) - $curl_time;

	if ($curl_time > $vbulletin->options['stc_curl_timeout']) {
		stc_log_action("curl time was: ".($curl_time)." and is higher than curl timeout: ".$vbulletin->options['stc_curl_timeout']);
	}
	
	DEVDEBUG("stc_profiling[]= -- fetched steaminfo in $curl_time seconds inside of the following request frame:");

	// It may happen every now and then that steam openid is too busy and sends you a 500 status.
    if ($http_response_header['http_code'] != '200' && $curl_errno > 0) {
		$msg = "Steam API seems to be unavaliable at this time. http-code: $http_response_header[http_code], curl error [$curl_errno]: $curl_error";
    	if (can_moderate()) trigger_error($msg, E_USER_NOTICE);
		stc_log_action($msg);
		DEVDEBUG("stc_profiling[] = Time: ".(microtime(1)-$time).", ret: http500");
    	return false;
	}

	$steam_info = array();
    $json_decoded = json_decode($raw_json);

    foreach ($json_decoded->response->players as $player) {

        $steam_info['steamid'] = $player->steamid;
        $steam_info['communityvisibilitystate'] = $player->communityvisibilitystate;
        $steam_info['personaname'] = $player->personaname;
        $steam_info['profileurl'] = $player->profileurl;
        $steam_info['avatar'] = $player->avatar;
        $steam_info['avatarfull'] = $player->avatarfull;
        $steam_info['avatarmedium'] = $player->avatarmedium;

    }

	// If we injected additional fields, it might happen that we hit some fields, which aren't necessarily valid.
	// Therefore it may happen that we get less actual fields than the whitelist contains. This ensures that
	// in this case we get a valid function response nontheless.
	// If we have no additional fields, then the response field count must be equal the whitelist otherwise
	// something is wrong with steam...

	$stc_cache->store($steam_id, $steam_info);
	DEVDEBUG("stc_profiling[] = Time: ".(microtime(1)-$time).", ret: http200");
	return $steam_info;
}

/**
 * Fetches a memberlist of the specified group
 * @param string $group_id the group id (also known as short tag)
 * @param int $limit limits the number of pages to retrieve from the group API. Huge groups are split up in chunks of 1000 steam ids. If you only need the basic info of a group
 * @param bool $ignore_cache allows to ignore any cached information and force to use fresh data
 * @param bool $verbose if true, prints additional output to std::out indicating the progress of fetching the memberlist.
 * use a limit of 1. If you wish to get the complete memberlist, set limit to 0. Keep in mind that this means a whole lot of queries if the group exceeds 5k members and therefore
 * slowing down the process significantly.
 * @return array|boolean array with contents of the specified fields from the query or false if the query was unsuccessfull because the group id wasn't found or the server didn't respond.
 */
function fetch_group_info($group_id, $limit=0, $ignore_cache=false, $verbose=false) {
	global $vbulletin;
	$stc_cache = stc_cache::getInstance();

	$time = microtime(1);

	if (empty($group_id)) {
		stc_log_action("fetch_group_info called with empty group_id!");
		return false;
	}

	if (!$ignore_cache) {
		$cached_data = $stc_cache->fetch("group_$group_id");
		if ($cached_data) {
			DEVDEBUG("stc_profiling[] = Time: ".(microtime(1)-$time).", ret: cache");
			return $cached_data;
		}
	}

	$steam_ids = array();
	$group_info = false;
	$page = 1;
	$pageCount = '?';
	if ($verbose) {
		echo "<br />Fetching max. <strong>$limit</strong> pages for group $group_id:<br />";
		vbflush();
	}
	while (1) {

		if ($verbose) {
			echo "- page <strong>$page</strong> of <strong>".min($limit, $pageCount)."</strong><br />";
			vbflush();
		}

		$url = sprintf("http://steamcommunity.com/groups/%s/memberslistxml?p=%d",
			$group_id,
			$page);

		$curl_time = microtime(1);
		// Needs curl to actually be able to pull data from steam api because of the special response-behaviour.
	    $ch = curl_init();
	    curl_setopt( $ch, CURLOPT_URL, $url );
	    @curl_setopt( $ch, CURLOPT_FOLLOWLOCATION, true );
	    curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
	    curl_setopt( $ch, CURLOPT_AUTOREFERER, true );
	    curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, false );
	    curl_setopt( $ch, CURLOPT_MAXREDIRS, 5 );
	    curl_setopt( $ch, CURLOPT_CONNECTTIMEOUT, $vbulletin->options['stc_curl_timeout'] ); #rc1: helps with page delay if steam is down
	    $raw_xml = curl_exec( $ch );
	    $http_response_header = curl_getinfo( $ch );
		$curl_errno = curl_errno($ch);
		$curl_error = curl_error($ch);
		curl_close($ch);

		$curl_time = microtime(1) - $curl_time;

		if ($curl_time > $vbulletin->options['stc_curl_timeout']) {
			stc_log_action("curl time was: ".($curl_time)." and is higher than curl timeout: ".$vbulletin->options['stc_curl_timeout']);
		}

		DEVDEBUG("stc_profiling[]= -- fetched groupinfo page #$page in $curl_time seconds");

		// It may happen every now and then that steam openid is too busy and sends you a 500 status.
	    if ($http_response_header['http_code'] != '200' && $curl_errno > 0) {
			$msg = "Steam API seems to be unavaliable at this time. http-code: $http_response_header[http_code], curl error [$curl_errno]: $curl_error";
	    	if (can_moderate()) trigger_error($msg, E_USER_NOTICE);
			stc_log_action($msg);
			DEVDEBUG("stc_profiling[] = Time: ".(microtime(1)-$time).", ret: http500");
	    	return false;
		}

		$old_err = error_reporting(0);
		try {
	   		if (empty($raw_xml) || !($xml = new SimpleXMLElement($raw_xml)) || !isset($xml->groupID64)) {
				DEVDEBUG("stc_profiling[] = Time: ".(microtime(1)-$time).", ret: http404");
				error_reporting($old_err);
				return false;
			}
		} catch (Exception $e) {
			$msg = "Steam API returned malformed XML for group query \"$group_id\": ".$e->getMessage();
			stc_log_action($msg);
			error_reporting($old_err);
			return false;
		}
		error_reporting($old_err);

		$pageCount = (int)$xml->totalPages;
		$memberCount = (int)$xml->memberCount;
		$memberOffset = (int)$xml->startingMember;
		$members = (array)$xml->members;

		if ($group_info === false) {
			$group_info = array(
			 	'groupID64' => "".$xml->groupID64,
			 	'memberCount' => $memberCount,
			 	'totalPages' => $pageCount,
			 	'members' => NULL
			);
		}

		$i = $memberOffset;
		foreach ($members['steamID64'] AS $mm) {
			$steam_ids[$i] = $mm;
			$i++;
		}

		if ($page >= $pageCount || ($limit && $page >= $limit)) {
			break;
		} else {
			$page++;
		}
	}

	$group_info['members'] = $steam_ids;

	$stc_cache->store("group_$group_id", $group_info);
	DEVDEBUG("stc_profiling[] = Time: ".(microtime(1)-$time).", ret: http200");
	return $group_info;
}

/**
 * Generates the steam id code representation in the form STEAM_0:0:12345 from a numerical steam id representation
 * @param mixed the numerical steam friend id value (note, that it should not be an integer, since integer values
 * can't handle a number > 2^31, whereas steam friend ids can be higher than 2^31.
 * @return string the STEAM_0:0:12345 representation
 */
function calc_steam_code($steam_id) {
	$srv = (substr($steam_id,-1)%2==0) ? 0 : 1;
	$ath = (substr($steam_id, -13)-1197960265728-$srv)/2;
	return 'STEAM_0:'.$srv.':'.$ath;
}

/**
 * Checks if the given steamid exists and is assigned to a forum user
 * @param array the array with the steamid as field
 * @return bool weather an user with this steam id has been found in the database
 */
function verify_steam_authentication($steam_info) {
	global $vbulletin;

	$userinfo = array();

	// return false if not a known steamid, or there is no associated user record
	if (empty($steam_info['steamid']) || !is_numeric($steam_info['steamid']))
	{
		return false;
	}
	if (!$userinfo = $vbulletin->db->query_first("
		SELECT userid, usergroupid, membergroupids, infractiongroupids, username, password, salt
		FROM " . TABLE_PREFIX . "user
		WHERE steam_link LIKE '".$vbulletin->db->escape_string($steam_info['steamid'])."|%'
	"))
	{
		return false;
	}

	// steam login successful, fetch hook and return true
	$return_value = true;
	return $return_value ? $userinfo : false;
}

/**
 * checks if the provided steamid is banned
 * @param array the array containing the steam id in the 'steamid' field
 * @return bool weather the steam id is banned or not
 */
function is_banned_steam_id($steam_info) {
	global $vbulletin;

	$banned_ids_raw = explode("\n", $vbulletin->options['stc_banned_steamids']);
	$banned_ids = array(
	 	'steam64'	=> array(),
	 	'friendid'	=> array(),
	 	'commid'	=> array()
	);
	foreach ($banned_ids_raw AS $bb) {
		$bb = trim($bb);
    	if (is_numeric($bb))
    		$banned_ids['friendid'][] = $bb;
    	elseif (preg_match('/STEAM_0:\d:\d+/', $bb) > 0)
    		$banned_ids['steam64'][] = $bb;
    	else
    		$banned_ids['commid'][] = $bb;
	}
	unset($banned_ids_raw);

	if (!empty($banned_ids['friendid']) && in_array($steam_info['steamid'], $banned_ids['friendid']))
		return true;

	$steam64 = calc_steam_code($steam_info['steamid']);
	if (!empty($banned_ids['steam64']) && in_array($steam64, $banned_ids['steam64']))
		return true;

	$steam_data = fetch_steam_info($steam_info['steamid'], $vbulletin->options['stc_apikey']);
	$profileurlbits = explode('/', $steam_data['profileurl']);
	$commid = $profileurlbits[count($profileurlbits)-2];
	unset($steam_data, $profileurlbits);
	if ($commid != $steam_info['steamid'] && !empty($banned_ids['commid']) && in_array($commid, $banned_ids['commid'])) {
		return true;
	}

	// else
	return false;
}

/**
 * updates all user with the correct usergroup depending on the steamgroup
 * @param int how many query entries to skip
 * @param int how many query entries to process per call
 * @param int sets a userid the query should begin with
 * @param bool use echo output
 * @cacheburst bool ignore cached group data, pull fresh data instead
 * @return int the userid of the last user which has been updated
 */
function update_user_steamgroup_assoc($limit_start, $limit_count, $start_at_userid=1, $verbose = true, $cacheburst = true) {
	global $vbulletin, $vbphrase;

	$db = &$vbulletin->db;

    if ($verbose) {
		echo '<p>' . $vbphrase['stc_reassigngroups_head'] . "<br />\n";
	    vbflush();
	}

    if ($vbulletin->options['stc_secondary_group_usergroup'] && $vbulletin->options['stc_secondary_group_id']) {
        if ($verbose) {
			echo $vbphrase['stc_fetchinggroupinfo'] . ' ... ';
	        vbflush();
	    }
        $group = fetch_group_info($vbulletin->options['stc_secondary_group_id'], $vbulletin->options['stc_fetch_max_grouppages'], $cacheburst, $verbose);
        if ($verbose) echo $vbphrase['done'] . "<br />\n";
        $testgroups = true;
    } else {
        $testgroups = false;
    }

    $users = $db->query_read_slave("
        SELECT userid, usergroupid, membergroupids, displaygroupid, steam_link
        FROM `" . TABLE_PREFIX . "user`
        WHERE userid > " . ($startat-1) . " AND steam_link <> ''
		ORDER BY userid
		".(isset($limit_start, $limit_count) ? "LIMIT $limit_start, $limit_count" : ""));

    if ($verbose) {
	    echo $vbphrase['stc_users_queried'] . '</p><p>';
	    vbflush();
	}

    $finishat = $vbulletin->GPC['startat'];

    while ($user = $db->fetch_array($users))
    {
        list($steam_id, $no_email, $no_passwd) = explode('|',$user['steam_link']);
        $steam_only = $no_email || $no_passwd;
        $changed = false;

        $oldprimary = $user['usergroupid'];
        $oldsecondary = $user['membergroupids'];

        // Check primary groupid
        if ($steam_only
            && $vbulletin->options['stc_default_usergroup']
            && $user['usergroupid'] != $vbulletin->options['stc_default_usergroup']
            && !in_array($user['usergroupid'], array(6,7,5,8))) // don't change usergroup if this user is an administrator, supermod, moderator or banned user
        {
            if ($user['usergroupid'] == $user['displaygroupid'] || $vbulletin->options['stc_set_primary_as_display']) {
                $user['displaygroupid'] = $vbulletin->options['stc_default_usergroup'];
            }
            $user['usergroupid'] = $vbulletin->options['stc_default_usergroup'];
            $changed = true;
        }

        // Check secondary groupid
        if ($vbulletin->options['stc_secondary_usergroup']) {
            $tmp = ensure_secondary_group_membership($user['membergroupids'], $vbulletin->options['stc_secondary_usergroup']);
			if ($tmp != $user['membergroupids']) {
				$user['membergroupids'] = $tmp;
				$changed = true;
			}
        }

        // Check groups
        if ($testgroups && $group !== false) {
            if (in_array($steam_id, $group['members'])) {
				$tmp = ensure_secondary_group_membership($user['membergroupids'], $vbulletin->options['stc_secondary_group_usergroup']);
                if ($tmp != $user['membergroupids']) {
                    $user['membergroupids'] = $tmp;
                    $changed = true;
                }
            } else {
				$tmp = remove_secondary_group_membership($user['membergroupids'], $vbulletin->options['stc_secondary_group_usergroup']);
                if ($tmp != $user['membergroupids']) {
                    $user['membergroupids'] = $tmp;
                    $changed = true;
                }
			}
        }

        if($changed)
        {
			$txt = construct_phrase($vbphrase['processing_x'], $user['userid']) .
	            ' Primary: ('.$oldprimary.'->'.$user['usergroupid'].'), Secondary: ('.$oldsecondary.'->'.$user['membergroupids'].')';
			stc_log_action($txt);
            if ($verbose) {
				echo "$txt ... ";
	            vbflush();
	        }

            $db->query_write("
                UPDATE `" . TABLE_PREFIX . "user`
                SET `usergroupid` = " . $user['usergroupid'] . ",
                    `displaygroupid` = " . $user['displaygroupid'] . ",
                    `membergroupids` = '" . $user['membergroupids'] . "'
                WHERE `userid` = '$user[userid]'
            ");

            if ($verbose) {
	            echo $vbphrase['done'] . "<br />\n";
	            vbflush();
	        }
        } elseif ($verbose) {
            echo "Userid: $user[userid]: ". $vbphrase['stc_no_changes_made'] . "<br />\n";
            vbflush();
        }

        $finishat = ($user['userid'] > $finishat ? $user['userid'] : $finishat);
    }

    return $finishat;
}


function ensure_secondary_group_membership($str, $groupid) {
	$tmp = empty($str) ? array() : explode(',', $str);
	$tmp[] = $groupid;
	$arr = array();
	for ($i=0,$n=count($tmp); $i<$n; ++$i) {
		if (!empty($tmp[$i]) && is_numeric($tmp[$i]) && $tmp[$i] > 0) {
			$arr[] = (int)$tmp[$i];
		}
	}
	$arr = array_unique($arr, SORT_NUMERIC);
	$tmp = implode(',',$arr);
	return $tmp;
}

function remove_secondary_group_membership($str, $groupid) {
	$tmp = empty($str) ? array() : explode(',', $str);
	$arr = array();
	for ($i=0,$n=count($tmp); $i<$n; ++$i) {
		if (!empty($tmp[$i]) && is_numeric($tmp[$i]) && $tmp[$i] > 0 && $tmp[$i] != $groupid) {
			$arr[] = (int)$tmp[$i];
		}
	}
	$arr = array_unique($arr, SORT_NUMERIC);
	$tmp = implode(',',$arr);
	return $tmp;
}


/**
 * Writes a line into the log file if a log path was specified
 * @param string the line to write. Note, that the line break will be added by the function
*/
function stc_log_action($message) {
	global $stc_loghandle, $vbulletin;

	if (empty($vbulletin->options['stc_logfile_path'])) return;

	if ($stc_loghandle == NULL) {
		$stc_loghandle = fopen($vbulletin->options['stc_logfile_path'], "a");
	}
	$ts = "[". date($vbulletin->options['logdateformat']) . "] ";
	fputs($stc_loghandle, $ts . $message . "\n");
}

?>